/************************************************************************/
/* C drivers for Intel's byte-wide FlashFile memory family              */
/************************************************************************/

/************************************************************************/
/* Copyright Intel Corporation, 1996                                    */
/* Example C Routines for Intel's byte-wide FlashFile memory family     */
/* File: ff_drv.c                                                       */
/* Author: Ken McKee                                                    */        
/*                                                                      */
/* Revision 1.2, September 13, 1996    Added S3/S5 suuport              */
/* Revision 1.1, July 17, 1996         Added SA/L and SC support        */ 
/* Revision 1.0, January 1, 1996       Initial release                  */
/*                                                                      */
/* NOTE: Commands can be written to any address in the                  */
/* block or device to be affected by that command.                      */        
/************************************************************************/

#include <stdio.h>
#include "stddefs.h"

void set_pin(char level)
{
/* set_pin is an implementation-dependent function which sets RP#          */
}

char block_erase(char *address)
{
	/* This procedure erases a 64K byte block on the 28F008SA             */
	/* and 28F0xxS3/S5/SC.                                                */
	
	char SR;                           /* SR variable returns SRD         */
	*address = 0x20;                   /* Single Block Erase command      */
	*address = 0xD0;                   /* Confirm command                 */
	while(!(BIT_7 & *address))         /* Poll SR until SR.7 = 1          */
	{
		/* Erase may be suspended here to                                */
		/* read or progra to a different block.                          */
	};
	SR = *address;                     /* Save SR before clearing it.     */
	*address = 0x50;                   /* Clear SR command and place into */
								/* read mode.                      */
	return(SR);                        /* Return SR to be checked for     */
								/* status of operation.            */
}

 
char program(char *address, char data)
{
	/* This procedure programs a byte to the 28F008SA and 28F0xxS3/S5/SC.   */

	char SR;                           /* SR variable returns SRD         */
	*address = 0x40;                   /* Progra command.                 */
	*address = data;                   /* Actual data write to flash      */
	while(!(BIT_7 & *address))         /* Poll SR until SR.7 = 1.         */
	{
		/* Program may be suspended here to read from a different        */
		/* location.                                                     */
	};
	SR = *address;                     /* Save SR before clearing it.     */
	*address = 0x50;                   /* Clear SR command and place into */
								/* read mode.                      */
	return(SR);                        /* Return SR to be checked         */
								/* for status of operation.        */
}

 
void erase_suspend_to_read(char *address, char *result)
{
	/* This procedure suspends an erase operation to do a read.           */
	/* Assume erase is underway. This function works for the 28F008SA     */
	/* and 28F0xxS3/S5/SC.                                                */
	
	*address = 0xB0;                   /* Erase Suspend command           */
	while(!(BIT_7 & *address));        /* Poll SR until SR.7 = 1          */
	*address = 0xFF;                   /* Read Flash Array command        */
	*result = *address;                /* Do the actual read. Any number  */
								/* of reads can be done here.      */
	*address = 0x70;                   /* Read SR command.                */
	if (BIT_6 & *address)              /* If SR.6 = 1 (erase incomplete)  */
		*address = 0xD0;              /* Erase Resume command            */
}
 
void erase_suspend_to_program(char *address, char data)
{
	/* This procedure suspends an erase operation to do a                 */
	/* program. Assume erase is underway. This function works for         */
	/* the 28F0xxS3/S5/SC.                                                */
	
	*address = 0xB0;                   /* Erase Suspend command           */
	while(!(BIT_7 & *address));        /* Poll SR until SR.7 = 1          */
	*address = 0x40;                   /* Program command.                */
	*address = data;                   /* Actual data write to flash.     */
	while(!(BIT_7 & *address))         /* Poll SR until SR.7 = 1.         */        
	{                                  
		/* Program may be suspended here to read from a different        */
		/* location.                                                     */
	};
	if (BIT_6 & *address)              /* If SR.6 = 1 (erase incomplete)  */
		*address = 0xD0;              /* Erase Resume command            */
								/* for status of operation.        */
}

void byte_suspend_to_read(char *address, char *result)
{
	/* This procedure suspends an program operation to do a read.         */
	/* Assume erase is underway. This function works for the              */
	/* 28F0xxS3/S5/SC.                                                    */
	
	*address = 0xB0;                   /* Erase Suspend command           */
	while(!(BIT_7 & *address));        /* Poll SR until SR.7 = 1          */
	*address = 0xFF;                   /* Read Flash Array command        */
	*result = *address;                /* Do the actual read. Any number  */
								/* of reads can be done here.      */
	*address = 0x70;                   /* Read array command.             */
	if (BIT_2 & *address)              /* If SR.2 = 1 (program            */
								/* incomplete)                     */
		*address = 0xD0;              /* Erase Resume command            */
}

char set_block_lock_bit(char *lock_address)
{
	/* This procedure locks a block on the 28F0xxS3/S5/SC.                */

	char SR;                           /* SR variable returns SRD.        */
					
	/* If the Master lock-bit is set, RP# = Vhh                           */
	/* set_pin(1);      */             /* Enable high voltage on to RP    */


	*lock_address = 0x60;              /* Set Block Lock-Bit command.     */
	*lock_address = 0x01;              /* Confirmation                    */
	while (!(BIT_7 & *lock_address));  /* Poll SR until SR.7 = 1          */
	
	/* If the Master lock-bit is set return RP# to Vih.                   */
	/* set_pin(0);      */             /* Disable high voltage on RP#     */
					
	SR = *lock_address;                /* Save SR before clearing it.     */
	*lock_address = 0x50;              /* Clear SR command and place into */            
								/* read mode.                      */
	return(SR);                        /* Return SR to be checked         */
								/* for status of operation.        */
}

char set_master_lock_bit(char *lock_address)
{
	/* This procedure locks a block on the 28F0xxS3/S5/SC.                */

	char SR;                           /* SR variable returns SRD         */
	set_pin(1);                        /* Enable high voltage on RP#      */
	*lock_address = 0x60;              /* Set Master Lock-Bit command.    */
	*lock_address = 0xF1;              /* Confirmation                    */
	while (!(BIT_7 & *lock_address));  /* Poll SR until SR.7 = 1          */
	set_pin(0);                        /* Disable high voltage on RP#     */
	SR = *lock_address;                /* Save SR before clearing it.     */
	*lock_address = 0x50;              /* Clear SR command and place into */           
								/* read mode.                      */
	return(SR);                        /* Return SR to be checked         */
								/* for status of operation.        */
}
 
char clear_block_lock_bits(char *lock_address)
{
	/* This procedure clears all block lock-bits on the 28F0xxS3/S5/SC.   */

	char SR;                           /* SR variable returns SRD         */
					
	/* If the Master lock-bit is set, RP# = Vhh                           */
	/* set_pin(1);      */             /* Enable high voltage on RP#      */

	*lock_address = 0x60;              /* Clear Block Lock-Bits command.  */
	*lock_address = 0xD0;              /* Confirmation                    */
	while (!(BIT_7 & *lock_address));  /* Poll SR until SR.7 = 1          */
	
	/* If the Master lock-bit was set return RP# to Vih.                  */ 
	/* set_pin(0);      */             /* Disable high voltage on RP#     */
				    
	SR = *lock_address;                /* Save SR before clearing it.     */
	*lock_address = 0x50;              /* Clear SR command and place into */
								/* read mode.                      */
	return(SR);                        /* Return SR to be checked         */
								/* for status of operation.        */
}

char read_identifier_codes(char *address)
{
	/* This procedure provides access to the 28F008SA's and               */
	/ * 28F0xxS3/S5/SCs'                                                  */
	/*      Manifactor Code                                               */
	/*      Device Code                                                   */
	/*      Master/Block Lock Configuration Code  ** 28F0xxS3/S5/SC only  */
	
	char code;                         /* Code variable returned          */ 

	*address = 0x90;                   /* Read Identifier Codes command   */
	code = *address;                   /* Store code.                     */
	return(code);                      /* Return value.                   */
}


char SR_full_status_check(char SR)
{
	/* This procedure performs a ful SR check. It is valid for            */
	/* program, block erase, lock-bet set and block lock-bit              */
	/* reset operations. This function works for the 28F008SA and         */
	/* 28F0xxS3/S5/SC.                                                    */
	/*                                                                    */
	/* Note: This procedure assumes that SR data resides in SR.           */
	/* This information is placed in the variable after the               */
	/* completion of each operation. If an error is detected, the         */
	/* previous operationshould be executed again.                        */

	char error_code;                   /* returns error code.             */

	if (SR & BIT_3)                    /* Vpp range error check.          */
		error_code = VPP_ERROR;       /* Set error code.                 */
	else if (SR & BIT_1)               /* Device protection error check.  */
		error_code = BLOCK_PROTECTION_ERROR;
								/* Set error code.                 */
	else if (SR & BIT_4){              /* Program error check.            */
		if (SR & BIT_5)               /* Command sequence error check.   */
			error_code = COMMAND_SEQ_ERROR;
								/* Set error code.                 */
		else                    
			error_code = PROGRAM_ERROR;
	}                                  /* Set program error code.         */
	else if (SR & BIT_5)               /* Block erase error check.        */
		error_code = ERASE_ERROR;     /* Set error code.                 */
	else                               /* No error detected.              */
		error_code = NO_ERROR;        /* Set error code.                 */
	return(error_code);                /* Return error code.              */
}

